#!/usr/bin/env node

/**
 * File Manifest Generator for Post Rapture Emergency Website
 * 
 * This utility scans the filesystem and generates a comprehensive JSON manifest
 * of all files and folders for the static file browser system.
 * 
 * Usage:
 *   node generate-manifest.js
 *   node generate-manifest.js --output custom-manifest.json
 *   node generate-manifest.js --path ./specific-folder
 *   node generate-manifest.js --verbose
 */

const fs = require('fs').promises;
const path = require('path');

class ManifestGenerator {
    constructor(options = {}) {
        // Default to project root (3 levels up from script location in webifyRK/search)
        const scriptDir = __dirname;
        const projectRoot = path.resolve(scriptDir, '..', '..', '..');
        this.rootPath = options.path || projectRoot;
        // Keep output file in the search directory
        this.outputFile = options.output || path.join(scriptDir, 'file-manifest.json');
        this.verbose = options.verbose || false;
        this.stats = {
            totalFiles: 0,
            totalFolders: 0,
            totalSize: 0,
            processedSections: 0,
            startTime: Date.now()
        };
        
        // User-facing file types only
        this.fileTypeMap = {
            // Documents
            '.pdf': { type: 'application/pdf', category: 'document', icon: '📄' },
            '.docx': { type: 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', category: 'document', icon: '📄' },
            '.pptx': { type: 'application/vnd.openxmlformats-officedocument.presentationml.presentation', category: 'document', icon: '📄' },
            '.txt': { type: 'text/plain', category: 'document', icon: '📄' },
            
            // Media
            '.mp4': { type: 'video/mp4', category: 'video', icon: '🎥' },
            '.mp3': { type: 'audio/mpeg', category: 'audio', icon: '🎵' },
            
            // Archives
            '.zip': { type: 'application/zip', category: 'archive', icon: '📦' }
        };
    }

    /**
     * Main execution method
     */
    async generate() {
        try {
            console.log('🚀 Starting file manifest generation...');
            console.log(`📁 Scanning directory: ${this.rootPath}`);
            
            // Generate the manifest
            const manifest = await this.buildManifest();
            
            // Write to file
            await this.writeManifest(manifest);
            
            // Display completion stats
            this.displayStats();
            
            console.log(`✅ Manifest generated successfully: ${this.outputFile}`);
            
        } catch (error) {
            console.error('❌ Error generating manifest:', error.message);
            if (this.verbose) {
                console.error(error.stack);
            }
            process.exit(1);
        }
    }

    /**
     * Check if a directory should be scanned
     */
    shouldScanDirectory(dirPath) {
        const relativePath = path.relative(this.rootPath, dirPath).replace(/\\/g, '/');
        
        // Exclude webifyRK directory completely
        if (relativePath.includes('webifyRK')) {
            return false;
        }
        
        return true;
    }

    /**
     * Check if a file should be included in the manifest
     * Only include user-facing content, exclude internal/development files
     */
    shouldIncludeFile(filePath) {
        const relativePath = path.relative(this.rootPath, filePath).replace(/\\/g, '/');
        const fileName = path.basename(filePath);
        const ext = path.extname(fileName).toLowerCase();
        
        // Exclude webifyRK directory (internal website files)
        if (relativePath.includes('webifyRK/')) {
            return false;
        }
        
        // Include _Read Me First.pdf in root
        if (fileName === '_Read Me First.pdf') {
            return true;
        }
        
        // Include Tutorial MP4 files from Section 01 (user-facing content)
        if (relativePath.startsWith('Section 01 - Materials for the Rapture Kit Sender/') && 
            fileName.startsWith('Tutorial') && ext === '.mp4') {
            return true;
        }
        
        // Include files with user-facing extensions
        return this.fileTypeMap.hasOwnProperty(ext);
    }

    // Alias for backward compatibility (temporary fix)
    shouldInclude(filePath) {
        return this.shouldIncludeFile(filePath);
    }

    /**
     * Build the complete manifest structure
     */
    async buildManifest() {
        const sections = await this.scanSections();
        
        const manifest = {
            generated: new Date().toISOString(),
            generator: 'Post Rapture Emergency Website Manifest Generator v1.0',
            rootPath: this.rootPath,
            totalSize: this.stats.totalSize,
            totalFiles: this.stats.totalFiles,
            totalFolders: this.stats.totalFolders,
            sections: sections,
            fileTypes: this.generateFileTypesSummary(sections),
            searchIndex: this.generateSearchIndex(sections)
        };

        return manifest;
    }

    /**
     * Scan all sections (Section 01, Section 02, etc.)
     */
    async scanSections() {
        const sections = [];
        const entries = await fs.readdir(this.rootPath, { withFileTypes: true });
        
        for (const entry of entries) {
            if (entry.isDirectory() && entry.name.startsWith('Section ')) {
                console.log(`📂 Processing section: ${entry.name}`);
                const section = await this.scanSection(entry.name);
                if (section) {
                    sections.push(section);
                    this.stats.processedSections++;
                }
            }
        }
        
        // Also scan root level files (like index.html)
        const rootSection = await this.scanRootFiles();
        if (rootSection.files.length > 0) {
            sections.unshift(rootSection); // Add at beginning
        }
        
        return sections;
    }

    /**
     * Scan root level files
     */
    async scanRootFiles() {
        const files = [];
        const entries = await fs.readdir(this.rootPath, { withFileTypes: true });
        
        for (const entry of entries) {
            if (entry.isFile() && this.shouldIncludeFile(path.join(this.rootPath, entry.name))) {
                const fileInfo = await this.getFileInfo(this.rootPath, entry.name);
                if (fileInfo) {
                    files.push(fileInfo);
                }
            }
        }
        
        return {
            id: 'root',
            name: 'Root Files',
            path: './',
            size: files.reduce((sum, file) => sum + (file.size || 0), 0),
            fileCount: files.length,
            folderCount: 0,
            folders: [],
            files: files
        };
    }

    /**
     * Scan a specific section directory
     */
    async scanSection(sectionName) {
        const sectionPath = path.join(this.rootPath, sectionName);
        
        try {
            const folders = [];
            const files = [];
            let totalSize = 0;
            let fileCount = 0;
            let folderCount = 0;
            
            const entries = await fs.readdir(sectionPath, { withFileTypes: true });
            
            for (const entry of entries) {
                const entryPath = path.join(sectionPath, entry.name);
                
                if (entry.isDirectory() && this.shouldScanDirectory(entryPath)) {
                    if (this.verbose) {
                        console.log(`  📁 Scanning folder: ${entry.name}`);
                    }
                    const folder = await this.scanFolder(sectionPath, entry.name);
                    if (folder) {
                        folders.push(folder);
                        totalSize += folder.size;
                        fileCount += folder.fileCount;
                        folderCount += 1 + folder.folderCount;
                    }
                } else if (entry.isDirectory() && this.verbose) {
                    console.log(`  ⏭️  Skipping directory: ${entry.name}`);
                } else if (entry.isFile() && this.shouldIncludeFile(entryPath)) {
                    const fileInfo = await this.getFileInfo(sectionPath, entry.name);
                    if (fileInfo) {
                        files.push(fileInfo);
                        totalSize += fileInfo.size;
                        fileCount += 1;
                    }
                }
            }
            
            this.stats.totalSize += totalSize;
            this.stats.totalFiles += fileCount;
            this.stats.totalFolders += folderCount;
            
            return {
                id: this.generateSectionId(sectionName),
                name: sectionName,
                path: `${sectionName}/`,
                size: totalSize,
                fileCount: fileCount,
                folderCount: folderCount,
                folders: folders,
                files: files
            };
            
        } catch (error) {
            console.error(`❌ Error scanning section ${sectionName}:`, error.message);
            return null;
        }
    }

    /**
     * Recursively scan a folder
     */
    async scanFolder(parentPath, folderName) {
        const folderPath = path.join(parentPath, folderName);
        
        try {
            const folders = [];
            const files = [];
            let totalSize = 0;
            let fileCount = 0;
            let folderCount = 0;
            
            const entries = await fs.readdir(folderPath, { withFileTypes: true });
            
            for (const entry of entries) {
                const entryPath = path.join(folderPath, entry.name);
                
                if (entry.isDirectory() && this.shouldScanDirectory(entryPath)) {
                    const subFolder = await this.scanFolder(folderPath, entry.name);
                    if (subFolder) {
                        folders.push(subFolder);
                        totalSize += subFolder.size;
                        fileCount += subFolder.fileCount;
                        folderCount += 1 + subFolder.folderCount;
                    }
                } else if (entry.isFile() && this.shouldIncludeFile(entryPath)) {
                    const fileInfo = await this.getFileInfo(folderPath, entry.name);
                    if (fileInfo) {
                        files.push(fileInfo);
                        totalSize += fileInfo.size;
                        fileCount += 1;
                    }
                }
            }
            
            const relativePath = path.relative(this.rootPath, folderPath).replace(/\\/g, '/') + '/';
            
            return {
                name: folderName,
                path: relativePath,
                size: totalSize,
                fileCount: fileCount,
                folderCount: folderCount,
                folders: folders,
                files: files
            };
            
        } catch (error) {
            if (this.verbose) {
                console.error(`⚠️  Error scanning folder ${folderName}:`, error.message);
            }
            return null;
        }
    }

    /**
     * Get detailed file information
     */
    async getFileInfo(parentPath, fileName) {
        const filePath = path.join(parentPath, fileName);
        
        try {
            const stats = await fs.stat(filePath);
            const ext = path.extname(fileName).toLowerCase();
            const typeInfo = this.fileTypeMap[ext] || { 
                type: 'application/octet-stream', 
                category: 'other', 
                icon: '📄' 
            };
            
            const relativePath = path.relative(this.rootPath, filePath).replace(/\\/g, '/');
            
            return {
                name: fileName,
                path: relativePath,
                size: stats.size,
                type: typeInfo.type,
                category: typeInfo.category,
                icon: typeInfo.icon,
                modified: stats.mtime.toISOString(),
                downloadable: true,
                sizeWarning: stats.size > 50 * 1024 * 1024, // 50MB warning
                sizeHuman: this.formatFileSize(stats.size)
            };
            
        } catch (error) {
            if (this.verbose) {
                console.error(`⚠️  Error reading file ${fileName}:`, error.message);
            }
            return null;
        }
    }

    /**
     * Generate search index for fast client-side searching
     */
    generateSearchIndex(sections) {
        const index = {
            files: [],
            folders: [],
            keywords: new Set()
        };
        
        const addToIndex = (item, type) => {
            const searchTerms = item.name.toLowerCase().split(/[\s\-_\.]+/);
            searchTerms.forEach(term => index.keywords.add(term));
            
            index[type].push({
                name: item.name,
                path: item.path,
                terms: searchTerms,
                size: item.size || 0,
                category: item.category || 'folder'
            });
        };
        
        const processFolder = (folder) => {
            addToIndex(folder, 'folders');
            folder.files?.forEach(file => addToIndex(file, 'files'));
            folder.folders?.forEach(subFolder => processFolder(subFolder));
        };
        
        sections.forEach(section => {
            addToIndex(section, 'folders');
            section.files?.forEach(file => addToIndex(file, 'files'));
            section.folders?.forEach(folder => processFolder(folder));
        });
        
        return {
            files: index.files,
            folders: index.folders,
            keywords: Array.from(index.keywords).sort()
        };
    }

    /**
     * Generate file types summary
     */
    generateFileTypesSummary(sections) {
        const summary = {};
        
        const processFiles = (files) => {
            files.forEach(file => {
                const category = file.category || 'other';
                if (!summary[category]) {
                    summary[category] = {
                        count: 0,
                        totalSize: 0,
                        extensions: new Set()
                    };
                }
                summary[category].count++;
                summary[category].totalSize += file.size || 0;
                summary[category].extensions.add(path.extname(file.name).toLowerCase());
            });
        };
        
        const processFolder = (folder) => {
            processFiles(folder.files || []);
            folder.folders?.forEach(subFolder => processFolder(subFolder));
        };
        
        sections.forEach(section => {
            processFiles(section.files || []);
            section.folders?.forEach(folder => processFolder(folder));
        });
        
        // Convert Sets to Arrays for JSON serialization
        Object.keys(summary).forEach(category => {
            summary[category].extensions = Array.from(summary[category].extensions);
            summary[category].totalSizeHuman = this.formatFileSize(summary[category].totalSize);
        });
        
        return summary;
    }

    /**
     * Generate section ID from section name
     */
    generateSectionId(sectionName) {
        return sectionName.toLowerCase()
            .replace(/\s+/g, '-')
            .replace(/[^a-z0-9\-]/g, '');
    }

    /**
     * Format file size in human readable format
     */
    formatFileSize(bytes) {
        if (bytes === 0) return '0 B';
        
        const k = 1024;
        const sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    /**
     * Write manifest to file
     */
    async writeManifest(manifest) {
        const manifestJson = JSON.stringify(manifest, null, 2);
        await fs.writeFile(this.outputFile, manifestJson, 'utf8');
        
        if (this.verbose) {
            console.log(`💾 Manifest written to: ${this.outputFile}`);
            console.log(`📊 Manifest size: ${this.formatFileSize(manifestJson.length)}`);
        }
        
        // Also generate JS fallback file for file:// protocol support
        await this.writeJSFallback(manifest);
    }
    
    /**
     * Write JS fallback file for file:// protocol
     */
    async writeJSFallback(manifest) {
        const jsFilePath = this.outputFile.replace('.json', '.js');
        const jsContent = `// Auto-generated file manifest for file:// protocol fallback
// Generated: ${new Date().toISOString()}
// DO NOT EDIT MANUALLY - This file is auto-generated by generate-manifest.js

window.FILE_MANIFEST = ${JSON.stringify(manifest, null, 2)};
`;
        
        await fs.writeFile(jsFilePath, jsContent, 'utf8');
        
        if (this.verbose) {
            console.log(`💾 JS fallback written to: ${jsFilePath}`);
        }
    }

    /**
     * Display generation statistics
     */
    displayStats() {
        const duration = Date.now() - this.stats.startTime;
        
        console.log('\n📊 Generation Statistics:');
        console.log(`   📁 Sections processed: ${this.stats.processedSections}`);
        console.log(`   📂 Total folders: ${this.stats.totalFolders.toLocaleString()}`);
        console.log(`   📄 Total files: ${this.stats.totalFiles.toLocaleString()}`);
        console.log(`   💾 Total size: ${this.formatFileSize(this.stats.totalSize)}`);
        console.log(`   ⏱️  Generation time: ${duration}ms`);
        console.log(`   🚀 Files/second: ${Math.round(this.stats.totalFiles / (duration / 1000))}`);
    }
}

/**
 * Command line interface
 */
function parseArguments() {
    const args = process.argv.slice(2);
    const options = {};
    
    for (let i = 0; i < args.length; i++) {
        switch (args[i]) {
            case '--output':
            case '-o':
                options.output = args[++i];
                break;
            case '--path':
            case '-p':
                options.path = args[++i];
                break;
            case '--verbose':
            case '-v':
                options.verbose = true;
                break;
            case '--help':
            case '-h':
                console.log(`
File Manifest Generator for Post Rapture Emergency Website

Usage:
  node generate-manifest.js [options]

Options:
  --output, -o <file>    Output file path (default: file-manifest.json)
  --path, -p <dir>       Root directory to scan (default: current directory)
  --verbose, -v          Enable verbose logging
  --help, -h             Show this help message

Examples:
  node generate-manifest.js
  node generate-manifest.js --output my-manifest.json --verbose
  node generate-manifest.js --path /path/to/content --output manifest.json
`);
                process.exit(0);
                break;
        }
    }
    
    return options;
}

/**
 * Main execution
 */
async function main() {
    const options = parseArguments();
    const generator = new ManifestGenerator(options);
    await generator.generate();
}

// Run if called directly
if (require.main === module) {
    main().catch(error => {
        console.error('💥 Fatal error:', error.message);
        process.exit(1);
    });
}

module.exports = ManifestGenerator;
